#!/bin/bash

# Zen Zone Spa Documentation Deployment Script
# This script helps deploy the documentation website to a cPanel hosting environment

set -e  # Exit on any error

echo "🌿 Zen Zone Spa Documentation Deployment Script"
echo "================================================"

# Colors for output
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Configuration
DOCS_HTML="docs.html"
ASSETS_DIR="backend-docs-assets"
README_FILE="docs-readme.md"

# Function to print colored output
print_status() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Check if we're in the right directory
check_environment() {
    print_status "Checking deployment environment..."
    
    if [ ! -f "$DOCS_HTML" ]; then
        print_error "docs.html not found! Please run this script from the backend directory."
        exit 1
    fi
    
    if [ ! -d "$ASSETS_DIR" ]; then
        print_error "backend-docs-assets directory not found!"
        exit 1
    fi
    
    print_success "Environment check passed ✓"
}

# Validate file structure
validate_files() {
    print_status "Validating file structure..."
    
    local errors=0
    
    # Check required files
    required_files=(
        "$DOCS_HTML"
        "$ASSETS_DIR/css/style.css"
        "$ASSETS_DIR/js/script.js"
        "$ASSETS_DIR/images/logo.png"
        "$ASSETS_DIR/images/spa-hero.jpg"
        "$ASSETS_DIR/images/services-overview.jpg"
        "$ASSETS_DIR/images/about-spa.jpg"
        "$ASSETS_DIR/images/favicon.png"
    )
    
    for file in "${required_files[@]}"; do
        if [ ! -f "$file" ]; then
            print_error "Missing required file: $file"
            errors=$((errors + 1))
        fi
    done
    
    if [ $errors -gt 0 ]; then
        print_error "Found $errors missing files. Please check your setup."
        exit 1
    fi
    
    print_success "File validation passed ✓"
}

# Create deployment package
create_package() {
    print_status "Creating deployment package..."
    
    # Create temporary deployment directory
    DEPLOY_DIR="zen-zone-docs-deployment"
    
    if [ -d "$DEPLOY_DIR" ]; then
        rm -rf "$DEPLOY_DIR"
    fi
    
    mkdir -p "$DEPLOY_DIR"
    
    # Copy files to deployment directory
    cp "$DOCS_HTML" "$DEPLOY_DIR/"
    cp -r "$ASSETS_DIR" "$DEPLOY_DIR/"
    
    # Copy documentation
    if [ -f "$README_FILE" ]; then
        cp "$README_FILE" "$DEPLOY_DIR/"
    fi
    
    # Create deployment instructions
    cat > "$DEPLOY_DIR/DEPLOYMENT_INSTRUCTIONS.txt" << EOF
Zen Zone Spa Documentation Deployment Instructions
=================================================

1. Upload to cPanel:
   - Upload docs.html to your public_html directory
   - Upload the entire backend-docs-assets folder to public_html
   
2. Final structure should be:
   public_html/
   ├── docs.html
   ├── backend-docs-assets/
   │   ├── css/
   │   ├── js/
   │   └── images/
   └── api/ (your existing API files)

3. Access your documentation at:
   https://yourdomain.com/docs.html

4. Customize:
   - Update domain name in the API documentation section
   - Modify contact information if needed
   - Replace images with your own if desired

For detailed instructions, see docs-readme.md

Support: zenzonerw@gmail.com | +250 730 000 016
EOF
    
    print_success "Deployment package created in $DEPLOY_DIR/ ✓"
}

# Create zip archive for easy upload
create_archive() {
    print_status "Creating zip archive for upload..."
    
    if command -v zip >/dev/null 2>&1; then
        cd "$DEPLOY_DIR"
        zip -r "../zen-zone-docs.zip" . -x "*.DS_Store" "*/.*"
        cd ..
        print_success "Archive created: zen-zone-docs.zip ✓"
        print_status "You can upload this zip file to your cPanel and extract it."
    else
        print_warning "zip command not found. Manual upload required."
        print_status "Upload the contents of $DEPLOY_DIR/ manually to your server."
    fi
}

# Validate HTML and CSS (if tools are available)
validate_code() {
    print_status "Running code validation (if tools available)..."
    
    # Check if HTML validator is available
    if command -v tidy >/dev/null 2>&1; then
        if tidy -q -e "$DOCS_HTML" >/dev/null 2>&1; then
            print_success "HTML validation passed ✓"
        else
            print_warning "HTML validation found issues (non-critical)"
        fi
    fi
    
    # Check CSS syntax (basic check)
    if [ -f "$ASSETS_DIR/css/style.css" ]; then
        if grep -q "syntax error" "$ASSETS_DIR/css/style.css" 2>/dev/null; then
            print_warning "Potential CSS syntax issues detected"
        else
            print_success "CSS basic validation passed ✓"
        fi
    fi
}

# Performance check
check_performance() {
    print_status "Checking file sizes for performance..."
    
    # Check image sizes
    for img in "$ASSETS_DIR"/images/*; do
        if [ -f "$img" ]; then
            size=$(du -h "$img" | cut -f1)
            basename_img=$(basename "$img")
            if [[ "$size" =~ ^[0-9]+M ]]; then
                size_num=$(echo "$size" | sed 's/M//')
                if [ "$size_num" -gt 2 ]; then
                    print_warning "Large image detected: $basename_img ($size) - consider optimizing"
                fi
            fi
        fi
    done
    
    # Check CSS size
    css_size=$(du -h "$ASSETS_DIR/css/style.css" | cut -f1)
    print_status "CSS size: $css_size"
    
    # Check JS size
    js_size=$(du -h "$ASSETS_DIR/js/script.js" | cut -f1)
    print_status "JavaScript size: $js_size"
    
    print_success "Performance check completed ✓"
}

# Main deployment function
main() {
    echo
    print_status "Starting deployment process..."
    echo
    
    # Run all checks and create package
    check_environment
    validate_files
    validate_code
    check_performance
    create_package
    create_archive
    
    echo
    print_success "🎉 Deployment package ready!"
    echo
    echo "Next steps:"
    echo "1. Upload zen-zone-docs.zip to your cPanel File Manager"
    echo "2. Extract the zip file in your public_html directory"
    echo "3. Access your documentation at https://yourdomain.com/docs.html"
    echo
    echo "For detailed instructions, see DEPLOYMENT_INSTRUCTIONS.txt"
    echo
    print_status "Happy deploying! 🚀"
}

# Run main function
main "$@" 