<?php
/**
 * Expense Model for Zen Zone Spa API
 * Represents a business expense
 */

class Expense {
    public $id;
    public $category_id;
    public $description;
    public $amount;
    public $expense_date;
    public $payment_method;
    public $receipt_number;
    public $notes;
    public $is_recurring;
    public $created_by;
    public $created_at;
    public $updated_at;
    
    // Legacy fields for compatibility
    public $category;
    public $date;
    
    /**
     * Create Expense from database row
     */
    public static function fromDatabaseRow($row) {
        $expense = new self();
        
        // Modern database fields
        $expense->id = $row['id'] ?? null;
        $expense->category_id = $row['category_id'] ?? null;
        $expense->description = $row['description'] ?? '';
        $expense->amount = (float)($row['amount'] ?? 0);
        $expense->expense_date = $row['expense_date'] ?? null;
        $expense->payment_method = $row['payment_method'] ?? 'cash';
        $expense->receipt_number = $row['receipt_number'] ?? null;
        $expense->notes = $row['notes'] ?? null;
        $expense->is_recurring = (bool)($row['is_recurring'] ?? false);
        $expense->created_by = $row['created_by'] ?? null;
        $expense->created_at = $row['created_at'] ?? null;
        $expense->updated_at = $row['updated_at'] ?? null;
        
        // Legacy fields for mobile app compatibility
        $expense->category = $row['category'] ?? null;
        $expense->date = $row['date'] ?? $row['expense_date'] ?? null;
        
        return $expense;
    }
    
    /**
     * Convert to array for JSON response
     */
    public function toArray() {
        return [
            'id' => $this->id,
            'category_id' => $this->category_id,
            'description' => $this->description,
            'amount' => $this->amount,
            'expense_date' => $this->expense_date,
            'payment_method' => $this->payment_method,
            'receipt_number' => $this->receipt_number,
            'notes' => $this->notes,
            'is_recurring' => $this->is_recurring,
            'created_by' => $this->created_by,
            'created_at' => $this->created_at,
            'updated_at' => $this->updated_at,
            
            // Legacy fields for mobile app
            'category' => $this->category,
            'date' => $this->date,
        ];
    }
    
    /**
     * Validate expense data
     */
    public static function validate($data) {
        $errors = [];
        
        if (empty($data['description'])) {
            $errors['description'] = 'Description is required';
        } elseif (strlen($data['description']) > 255) {
            $errors['description'] = 'Description must be less than 255 characters';
        }
        
        if (empty($data['amount']) || !is_numeric($data['amount']) || $data['amount'] <= 0) {
            $errors['amount'] = 'Valid amount is required';
        }
        
        if (empty($data['date']) && empty($data['expense_date'])) {
            $errors['date'] = 'Expense date is required';
        }
        
        // Validate date format
        $date = $data['date'] ?? $data['expense_date'] ?? '';
        if ($date && !strtotime($date)) {
            $errors['date'] = 'Invalid date format';
        }
        
        return [
            'isValid' => empty($errors),
            'errors' => $errors
        ];
    }
    
    /**
     * Get valid payment methods
     */
    public static function getPaymentMethods() {
        return ['cash', 'card', 'bank_transfer', 'mobile_money'];
    }
    
    /**
     * Get default expense categories
     */
    public static function getDefaultCategories() {
        return [
            'Supplies',
            'Utilities', 
            'Maintenance',
            'Marketing',
            'Staff',
            'Rent',
            'Equipment',
            'Other'
        ];
    }
}
