<?php
/**
 * Massage Model for Zen Zone Spa API
 * Represents a massage session
 */

class Massage {
    public $id;
    public $massage_type_id;
    public $therapist_id;
    public $client_id;
    public $client_name;
    public $amount;
    public $duration;
    public $therapist_commission;
    public $net_income;
    public $session_date;
    public $status;
    public $payment_method;
    public $notes;
    public $rating;
    public $created_by;
    public $created_at;
    public $updated_at;
    
    // Payment tracking fields
    public $payment_sent_to_momo;
    public $payment_verified;
    public $payment_verified_at;
    public $payment_verified_by;
    public $payment_notes;
    
    // Legacy fields for compatibility with current mobile app
    public $type;
    public $therapist;
    public $date;
    
    /**
     * Create Massage from database row
     */
    public static function fromDatabaseRow($row) {
        $massage = new self();
        
        // Modern database fields
        $massage->id = $row['id'] ?? null;
        $massage->massage_type_id = $row['massage_type_id'] ?? null;
        $massage->therapist_id = $row['therapist_id'] ?? null;
        $massage->client_id = $row['client_id'] ?? null;
        $massage->client_name = $row['client_name'] ?? null;
        $massage->amount = (float)($row['amount'] ?? 0);
        $massage->duration = (int)($row['duration'] ?? 60);
        $massage->therapist_commission = (float)($row['therapist_commission'] ?? 0);
        $massage->net_income = (float)($row['net_income'] ?? 0);
        $massage->session_date = $row['session_date'] ?? null;
        $massage->status = $row['status'] ?? 'completed';
        $massage->payment_method = $row['payment_method'] ?? 'cash';
        $massage->notes = $row['notes'] ?? null;
        $massage->rating = $row['rating'] ?? null;
        $massage->created_by = $row['created_by'] ?? null;
        $massage->created_at = $row['created_at'] ?? null;
        $massage->updated_at = $row['updated_at'] ?? null;
        
        // Payment tracking fields
        $massage->payment_sent_to_momo = isset($row['payment_sent_to_momo']) ? (bool)$row['payment_sent_to_momo'] : false;
        $massage->payment_verified = isset($row['payment_verified']) ? (bool)$row['payment_verified'] : false;
        $massage->payment_verified_at = $row['payment_verified_at'] ?? null;
        $massage->payment_verified_by = $row['payment_verified_by'] ?? null;
        $massage->payment_notes = $row['payment_notes'] ?? null;
        
        // Legacy fields for mobile app compatibility
        $massage->type = $row['type'] ?? null;
        $massage->therapist = $row['therapist'] ?? null;
        $massage->date = $row['date'] ?? $row['session_date'] ?? null;
        
        return $massage;
    }
    
    /**
     * Convert to array for JSON response
     */
    public function toArray() {
        return [
            'id' => $this->id,
            'massage_type_id' => $this->massage_type_id,
            'therapist_id' => $this->therapist_id,
            'client_id' => $this->client_id,
            'client_name' => $this->client_name,
            'amount' => $this->amount,
            'duration' => $this->duration,
            'therapist_commission' => $this->therapist_commission,
            'net_income' => $this->net_income,
            'session_date' => $this->session_date,
            'status' => $this->status,
            'payment_method' => $this->payment_method,
            'notes' => $this->notes,
            'rating' => $this->rating,
            'created_by' => $this->created_by,
            'created_at' => $this->created_at,
            'updated_at' => $this->updated_at,
            
            // Payment tracking fields
            'payment_sent_to_momo' => $this->payment_sent_to_momo,
            'payment_verified' => $this->payment_verified,
            'payment_verified_at' => $this->payment_verified_at,
            'payment_verified_by' => $this->payment_verified_by,
            'payment_notes' => $this->payment_notes,
            
            // Legacy fields for mobile app
            'type' => $this->type,
            'therapist' => $this->therapist,
            'date' => $this->date,
        ];
    }
    
    /**
     * Calculate commission based on amount
     */
    public function calculateCommission($rate = 0.30) {
        return $this->amount * $rate;
    }
    
    /**
     * Calculate net income after commission
     */
    public function calculateNetIncome($rate = 0.30) {
        return $this->amount * (1 - $rate);
    }
    
    /**
     * Validate massage data
     */
    public static function validate($data) {
        $errors = [];
        
        if (empty($data['type']) && empty($data['massage_type_id'])) {
            $errors['type'] = 'Massage type is required';
        }
        
        if (empty($data['amount']) || !is_numeric($data['amount']) || $data['amount'] <= 0) {
            $errors['amount'] = 'Valid amount is required';
        }
        
        if (empty($data['therapist']) && empty($data['therapist_id'])) {
            $errors['therapist'] = 'Therapist is required';
        }
        
        if (empty($data['date']) && empty($data['session_date'])) {
            $errors['date'] = 'Session date is required';
        }
        
        return [
            'isValid' => empty($errors),
            'errors' => $errors
        ];
    }
}
